"use client"

import type React from "react"

import { useState } from "react"
import { useRouter } from "next/navigation"
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { ArrowLeft, Send, MoreVertical } from "lucide-react"
import { cn } from "@/lib/utils"

interface Message {
  id: string
  text: string
  sender: "me" | "other"
  timestamp: string
}

const MOCK_MESSAGES: Message[] = [
  {
    id: "1",
    text: "Hey! How are you doing?",
    sender: "other",
    timestamp: "7:30 pm",
  },
  {
    id: "2",
    text: "I'm doing great! Just working on some new features.",
    sender: "me",
    timestamp: "7:32 pm",
  },
  {
    id: "3",
    text: "That sounds exciting! What are you building?",
    sender: "other",
    timestamp: "7:33 pm",
  },
  {
    id: "4",
    text: "A new social media app called MyPowerPost. It's been a lot of fun!",
    sender: "me",
    timestamp: "7:35 pm",
  },
  {
    id: "5",
    text: "Wow! I'd love to check it out when it's ready.",
    sender: "other",
    timestamp: "7:36 pm",
  },
  {
    id: "6",
    text: "Thank you",
    sender: "me",
    timestamp: "7:36 pm",
  },
]

export default function ChatPage({ params }: { params: { id: string } }) {
  const router = useRouter()
  const [messages, setMessages] = useState<Message[]>(MOCK_MESSAGES)
  const [newMessage, setNewMessage] = useState("")

  const handleSend = () => {
    if (!newMessage.trim()) return

    const message: Message = {
      id: Date.now().toString(),
      text: newMessage,
      sender: "me",
      timestamp: new Date().toLocaleTimeString([], { hour: "numeric", minute: "2-digit" }),
    }

    setMessages([...messages, message])
    setNewMessage("")
  }

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault()
      handleSend()
    }
  }

  return (
    <div className="flex flex-col h-screen bg-background">
      <header className="sticky top-0 z-50 bg-card border-b border-border">
        <div className="flex items-center gap-3 h-16 px-4 max-w-2xl mx-auto">
          <Button variant="ghost" size="icon" onClick={() => router.back()}>
            <ArrowLeft className="w-5 h-5" />
          </Button>

          <Avatar className="w-10 h-10">
            <AvatarImage src="/placeholder.svg?key=8dj2k" />
            <AvatarFallback>MM</AvatarFallback>
          </Avatar>

          <div className="flex-1">
            <h2 className="font-semibold text-foreground">Marvin McKinney</h2>
            <p className="text-xs text-muted-foreground">Active now</p>
          </div>

          <Button variant="ghost" size="icon">
            <MoreVertical className="w-5 h-5" />
          </Button>
        </div>
      </header>

      <main className="flex-1 overflow-y-auto">
        <div className="max-w-2xl mx-auto p-4 space-y-4">
          {messages.map((message) => (
            <div key={message.id} className={cn("flex", message.sender === "me" ? "justify-end" : "justify-start")}>
              <div className={cn("flex gap-2 max-w-[70%]", message.sender === "me" && "flex-row-reverse")}>
                {message.sender === "other" && (
                  <Avatar className="w-8 h-8">
                    <AvatarImage src="/placeholder.svg?key=09sj3" />
                    <AvatarFallback>MM</AvatarFallback>
                  </Avatar>
                )}
                <div>
                  <div
                    className={cn(
                      "rounded-2xl px-4 py-2",
                      message.sender === "me"
                        ? "bg-primary text-primary-foreground rounded-br-sm"
                        : "bg-secondary text-foreground rounded-bl-sm",
                    )}
                  >
                    <p className="text-sm">{message.text}</p>
                  </div>
                  <p className={cn("text-xs text-muted-foreground mt-1 px-1", message.sender === "me" && "text-right")}>
                    {message.timestamp}
                  </p>
                </div>
              </div>
            </div>
          ))}
        </div>
      </main>

      <footer className="sticky bottom-0 bg-card border-t border-border">
        <div className="max-w-2xl mx-auto p-4">
          <div className="flex items-end gap-2">
            <Input
              placeholder="Type a message..."
              value={newMessage}
              onChange={(e) => setNewMessage(e.target.value)}
              onKeyPress={handleKeyPress}
              className="flex-1 min-h-[48px] max-h-32"
            />
            <Button onClick={handleSend} disabled={!newMessage.trim()} size="icon" className="h-12 w-12 shrink-0">
              <Send className="w-5 h-5" />
            </Button>
          </div>
        </div>
      </footer>
    </div>
  )
}
